import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:six_pos/common/widgets/custom_loader_widget.dart';
import 'package:six_pos/helper/responsive_helper.dart';
import 'package:six_pos/util/dimensions.dart';
import 'package:six_pos/util/styles.dart';

class SliverPaginatedListWidget extends StatefulWidget {
  final ScrollController scrollController;
  final Function(int? offset) onPaginate;
  final int? totalSize;
  final int? offset;
  final int? limit;
  final bool isDisableWebLoader;
  final Widget Function(Widget loaderWidget) builder;

  final bool enabledPagination;
  final bool reverse;
  const SliverPaginatedListWidget({
    super.key, required this.scrollController, required this.onPaginate, required this.totalSize,
    required this.offset, required this.builder, this.enabledPagination = true, this.reverse = false, this.limit = 10,
    this.isDisableWebLoader = false,
  });

  @override
  State<SliverPaginatedListWidget> createState() => _SliverPaginatedListWidgetState();
}

class _SliverPaginatedListWidgetState extends State<SliverPaginatedListWidget> {
  int? _offset;
  late List<int?> _offsetList;
  bool _isLoading = false;
  bool _isDisableLoader = true;

  @override
  void initState() {
    super.initState();

    _offset = 1;
    _offsetList = [1];

    widget.scrollController.addListener(() {
      if (widget.scrollController.position.pixels == widget.scrollController.position.maxScrollExtent
          && widget.totalSize != null && !_isLoading && widget.enabledPagination) {
        if(mounted && !ResponsiveHelper.isDesktop(context)) {
          _paginate();
        }
      }
    });
  }

  void _paginate() async {
    int pageSize = (widget.totalSize! / widget.limit!).ceil();
    if (_offset! < pageSize && !_offsetList.contains(_offset!+1)) {

      setState(() {
        _offset = _offset! + 1;
        _offsetList.add(_offset);
        _isLoading = true;
      });
      await widget.onPaginate(_offset);
      if(mounted) {
        setState(() {
          _isLoading = false;
        });
      }

    }else {
      if(_isLoading) {
        setState(() {
          _isLoading = false;
        });
      }
    }
  }

  @override
  Widget build(BuildContext context) {
    if(widget.offset != null) {
      _offset = widget.offset;
      _offsetList = [];
      for(int index=1; index<=widget.offset!; index++) {
        _offsetList.add(index);
      }
    }

    _isDisableLoader = (ResponsiveHelper.isDesktop(context)
        && (widget.totalSize == null
            || _offset! >= (widget.totalSize! / (widget.limit ?? 10)).ceil()
            || _offsetList.contains(_offset!+1)));


    return Column(children: [

      widget.reverse ? const SizedBox() : widget.builder(_LoadingWidget(
        onTap: _paginate,
        isLoading: _isLoading,
        totalSize: widget.totalSize,
        isDisabledLoader: _isDisableLoader,
      )),

      if(widget.isDisableWebLoader) _LoadingWidget(
        onTap: _paginate,
        isLoading: _isLoading,
        totalSize: widget.totalSize,
        isDisabledLoader: _isDisableLoader,
      ),

      widget.reverse ? widget.builder(_LoadingWidget(
        onTap: _paginate,
        isLoading: _isLoading,
        totalSize: widget.totalSize,
        isDisabledLoader: _isDisableLoader,
      )) : const SizedBox(),

    ]);
  }

}

class _LoadingWidget extends StatelessWidget {
  const _LoadingWidget({
    required  this.isLoading,
    required this.totalSize,
    required this.isDisabledLoader,
    required this.onTap,
  });

  final bool isLoading;
  final bool isDisabledLoader;
  final int? totalSize;
  final Function onTap;

  @override
  Widget build(BuildContext context) {
    return isDisabledLoader ?  SizedBox(
      height: ResponsiveHelper.isDesktop(context) ? Dimensions.paddingSizeLarge : 0,
    ) :  Center(child: Padding(
      padding: (isLoading || ResponsiveHelper.isDesktop(context)) ?  const EdgeInsets.all(Dimensions.paddingSizeDefault) : EdgeInsets.zero,
      child: isLoading ? CustomLoaderWidget() : (ResponsiveHelper.isDesktop(context) && totalSize != null) ? InkWell(
        onTap: ()=> onTap(),
        child: Container(
          width: 150,
          padding: const EdgeInsets.symmetric(
            vertical: Dimensions.paddingSizeSmall,
            horizontal: Dimensions.paddingSizeLarge,
          ),
          alignment: Alignment.center,
          decoration: BoxDecoration(
            borderRadius: BorderRadius.circular(Dimensions.radiusDefault),
            border: Border.all(color: Theme.of(context).primaryColor),
          ),
          child: Text('   ${'see_more'.tr}', style: ubuntuRegular.copyWith(
            fontSize: Dimensions.fontSizeLarge,
            color: Theme.of(context).primaryColor,
          )),

        ),
      ) : const SizedBox(),
    ));
  }
}